function piniaPersistPlugin({store}) {
	if (store.$id === "calendar" || store.$id === "maps" || store.$id === "stripe" || store.$id === "validationEmail") return;

	const savedState = localStorage.getItem(store.$id);
	if (savedState) {
		store.$patch(JSON.parse(savedState));
	}

	store.$subscribe((mutation, state) => {
		localStorage.setItem(store.$id, JSON.stringify(state));
	});
}

document.addEventListener("DOMContentLoaded", () => {
	const {loadModule, vueVersion} = window["vue3-sfc-loader"];
	const {createPinia} = Pinia;
	const {h} = Vue;
	const {createVuetify} = Vuetify;
	const pinia = createPinia();
	pinia.use(piniaPersistPlugin);

	const propertyData = window.propertyData;

	// Configuração personalizada do Vuetify
	const vuetify = createVuetify({
		components: Vuetify.components,
		directives: Vuetify.directives,
		theme: {
			defaultTheme: "sasLight",
			themes: {
				sasLight: {
					dark: false,
					colors: {
						primary: "#212020",
						secondary: "#84a015",
					},
				},
			},
		},
		display: {
			thresholds: {
				xs: 0,
				sm: 600,
				md: 960,
				lg: 1280,
				xl: 1920,
			},
			scrollBarWidth: 24,
		},
	});

	const options = {
		moduleCache: {
			vue: Vue,
			myData: {vueVersion},
		},
		async getFile(url) {
			const res = await fetch(url);
			if (!res.ok)
				throw Object.assign(new Error(res.statusText + " " + url), {
					res,
				});
			return {
				getContentData: (asBinary) =>
					asBinary ? res.arrayBuffer() : res.text(),
			};
		},
		addStyle(styleStr) {
			const style = document.createElement("style");
			style.textContent = styleStr;
			const ref = document.head.getElementsByTagName("style")[0] || null;
			document.head.insertBefore(style, ref);
		},
		log(type, ...args) {
			console.log(type, ...args);
		},
	};

	window.loadModule = loadModule;
	window.options = options;

	// Função de fábrica para criar uma nova instância do app
	const createAppInstance = () => {
		const vueRouter = VueRouter.createRouter({
			history: VueRouter.createWebHistory(
				`/${window.env?.site_url_path ?? ""}`,
			),
			routes: [
				{
					name: "guest",
					path: `/${window.env?.sas_guest_panel_route ?? "user"}`,
					component: () =>
						loadModule(
							`${window.SAS_URL}/src/components/page/guest/Booking.vue`,
							options,
						),
					meta: {title: "Bookings"},
				},
				{
					name: "guest.booking-view",
					path: `/${
						window.env?.sas_guest_panel_route ?? "user"
					}/booking/:id/:view?`,
					component: () =>
						loadModule(
							`${window.SAS_URL}/src/components/page/guest/BookingView.vue`,
							options,
						),
					meta: {title: "Booking"},
				},
				{
					name: "guest.settings",
					path: `/${
						window.env?.sas_guest_panel_route ?? "user"
					}/settings`,
					component: () =>
						loadModule(
							`${window.SAS_URL}/src/components/page/guest/Settings.vue`,
							options,
						),
					meta: {title: "Settings"},
				},
				// {
				// 		name: 'properties',
				// 		path: `/${window.env?.sas_route_vendor_property_main_featured ?? 'property'}`,
				// 		component: {template: '<page-property />'},
				// 		meta: {title: 'Property Search'},
				// },
				// {
				// 		name: 'properties.details',
				// 		path: `/${window.env?.sas_route_vendor_property_main_featured ?? 'property'}/details/:id/:slug`,
				// 		component: {template: '<page-property-details />'},
				// 		meta: {title: 'Property Details'},
				// },
				// {
				// 		name: 'properties.details.single',
				// 		path: `/${window.env?.sas_route_vendor_property_main_featured ?? 'property'}/details`,
				// 		component: {template: '<page-property-details />'},
				// 		meta: {title: 'Property Details'},
				// },
				{
					name: "recovery.password",
					path: `/recovery/:hash`,
					component: () =>
						loadModule(
							`${window.SAS_URL}/src/components/page/Recovery.vue`,
							options,
						),
					meta: {title: "Recovery Password"},
				},
				{
					name: "error-404",
					path: `/:pathMatch(.*)*`,
					component: () =>
						loadModule(
							`${window.SAS_URL}/src/components/AppEmpty.vue`,
							options,
						),
					meta: {title: "Page not found"},
				},
			],
		});
		const app = Vue.createApp({
			components: {
				"search-single": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/SearchSingle.vue`,
						options,
					),
				),
				calendar: Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/calendar/Calendar.vue`,
						options,
					),
				),
				"sas-logo": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/Logo.vue`,
						options,
					),
				),
				"banner-single": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/BannerSingle.vue`,
						options,
					),
				),
				"banner-multi": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/BannerMulti.vue`,
						options,
					),
				),
				"home-discover": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/Discover.vue`,
						options,
					),
				),
				"property-basic-information": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/BasicInformation.vue`,
						options,
					),
				),
				"property-house-rules": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/HouseRules.vue`,
						options,
					),
				),
				"property-holiday-types": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/HolidayTypes.vue`,
						options,
					),
				),
				"property-amenities": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/Amenities.vue`,
						options,
					),
				),
				"property-house-configuration": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/HouseConfiguration.vue`,
						options,
					),
				),
				"property-search-form-location": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/SearchFormLocation.vue`,
						options,
					),
				),
				"property-search-form": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/SearchForm.vue`,
						options,
					),
				),
				"property-availability": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/Availability.vue`,
						options,
					),
				),
				"guest-index": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/guests/index.vue`,
						options,
					),
				),
				"properties-index": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/properties/index.vue`,
						options,
					),
				),
				"property-details": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/property/details.vue`,
						options,
					),
				),
				"property-maps": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/AppMap.vue`,
						options,
					),
				),
				"recovery-password": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/page/Recovery.vue`,
						options,
					),
				),
				"list-properties-view-port": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/ListViewPort.vue`,
						options,
					),
				),
				"single-property-selection": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/property/SinglePropertySelection.vue`,
						options,
					),
				),
				"accessibility-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/AccessibilityFilter.vue`,
						options,
					),
				),
				"sustainability-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/SustainabilityFilter.vue`,
						options,
					),
				),
				"pets-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/PetsFilter.vue`,
						options,
					),
				),
				"pools-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/PoolsFilter.vue`,
						options,
					),
				),
				"houses-linen-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/HousesLinenFilter.vue`,
						options,
					),
				),
				"waterfront-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/WaterfrontFilter.vue`,
						options,
					),
				),
				"beachfront-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/BeachfrontFilter.vue`,
						options,
					),
				),
				"unique-accommodations-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/UniqueAccommodationsFilter.vue`,
						options,
					),
				),
				"ski-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/SkiFilter.vue`,
						options,
					),
				),
				"dedicated-workspace-filter": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/DedicatedWorkspaceFilter.vue`,
						options,
					),
				),
				"list-of-filters": Vue.defineAsyncComponent(() =>
					loadModule(
						`${window.SAS_URL}/src/components/shortcodes/home/filter/ListOfFilters.vue`,
						options,
					),
				),
			},
			data() {
				return {propertyData: propertyData};
			},
		});

		window.$moment = moment;

		function onLoadRecaptcha() {
			window.$recaptcha = async function (action) {
				if (typeof grecaptcha === "undefined") {
					console.error("grecaptcha não está disponível.");
					return null;
				}
				return await grecaptcha.execute(
					"6LeKQNwZAAAAAIyA1es6Dg0X_uC2qXjWjOC53I4D",
					{action},
				);
			};
		}

		const recaptchaScript = document.createElement("script");
		recaptchaScript.src =
			"https://www.google.com/recaptcha/api.js?render=6LeKQNwZAAAAAIyA1es6Dg0X_uC2qXjWjOC53I4D";
		recaptchaScript.async = true;
		recaptchaScript.defer = true;
		recaptchaScript.onload = onLoadRecaptcha;
		document.head.appendChild(recaptchaScript);

		app.use(window.$recaptcha);
		app.use(pinia);
		app.use(vuetify);
		app.use(vueRouter);
		app.directive("imask", {
			input: false,
			settings: {},
			imask: false,

			fireEvent(el, eventName, data) {
				let e = document.createEvent("CustomEvent");
				e.initCustomEvent(eventName, true, true, data);
				el.dispatchEvent(e);
			},

			getSettings(bindValue) {
				if (!bindValue) return false;
				let settings = bindValue;

				if (Array.isArray(bindValue)) {
					settings = {
						mask: bindValue.map((mask) => {
							return {mask};
						}),
					};
				}

				return settings;
			},

			beforeMount(el, bind, vnode, prevVnode) {
				// el.input = el.tagName=='INPUT' ? el : el.querySelector('input');
				// el.settings = bind.dir.getSettings(bind.value);
				// if (!el.settings) return;
				// el.imask = IMask(el.input, el.settings);
			},

			updated(el, bind, vnode, prevVnode) {
				el.input =
					el.input ||
					(el.tagName == "INPUT" ? el : el.querySelector("input"));
				el.settings = bind.dir.getSettings(bind.value);

				// Update
				if (el.imask) {
					if (el.settings) {
						el.imask.masked.value = el.input.value;
						el.imask.updateOptions(el.settings);
						el.imask._onChange();
					} else {
						el.imask.destroy();
					}
				}

				// Create
				else {
					if (el.settings) {
						el.imask = IMask(el.input, el.settings);
					}
				}
			},

			unmounted(el, bind, vnode, prevVnode) {
				if (!el.imask) return;
				el.imask.destroy();
			},
		});

		return app;
	};

	const ids = [
		"sas-logo",
		"banner-single",
		"banner-multi",
		"search-single",
		"home-discover",
		"sas-calendar",
		"property-basic-information",
		"property-house-rules",
		"property-holiday-types",
		"property-amenities",
		"property-house-configuration",
		"property-search-form-location",
		"property-search-form",
		"property-availability",
		"guest-index",
		"properties-index",
		"property-details-index",
		"property-maps-all",
		"property-maps-load",
		"recovery-password-index",
		"list-properties-view-port",
		"single-property-selection",
		"accessibility-filter",
		"sustainability-filter",
		"pets-filter",
		"pools-filter",
		"houses-linen-filter",
		"waterfront-filter",
		"beachfront-filter",
		"unique-accommodations-filter",
		"ski-filter",
		"dedicated-workspace-filter",
		"list-of-filters"
	];

	ids.forEach((id) => {
		const element = document.getElementById(id);
		if (element) {
			const appInstance = createAppInstance();
			appInstance.mount(`#${id}`);
		}
	});
});
