<?php

namespace Saswp;

use SAS_Plugin;
use Saswp\DTOs\Response\PropertyListDTO;
use Saswp\DTOs\SasSettingsDTO;
use WP_REST_Request;

class Settings {
	private SasSettingsDTO $sasSettingsDTO;
	private PropertyListDTO $propertyListDTO;

	private array $settings;

	public function save( WP_REST_Request $wpRestRequest ) {
		$defaults = $this->defaults();

		$_data = $wpRestRequest->get_body_params()['settings'] ?? [];

		$_data['sas_route_vendor_property_main_featured'] = $defaults['sas_route_vendor_property_main_featured'];
		$_data['sas_guest_panel_route']                   = $defaults['sas_guest_panel_route'];

		$version                   = get_option( 'sas_version_save' ) + 1;
		$_data['sas_version_save'] = $version;
		$sasSettingsDTO            = new SasSettingsDTO( $_data );

		// Force Wordpress settings default page options show_on_front && page_on_front
		// if (isset($data['sas_page_home']) and !empty($data['sas_page_home'])) {
		//     update_option('show_on_front', 'page');
		//     update_option('page_on_front', $data['sas_page_home']);
		// }

		foreach ( $sasSettingsDTO->toArray() as $key => $value ) {
			if ( ! isset( $defaults[ $key ] ) ) {
				continue;
			}
			update_option( $key, $value );
		}

		flush_rewrite_rules();
		SAS_Plugin::write_file();
        SAS_Plugin::update_sas_config($sasSettingsDTO);

		return self::all( [ 'success' => true ] );
	}

	private function defaults() {
		return [
			'sas_api_token'                           => '',
			'sas_api_url'                             => 'https://sys.searchandstay.com',
			'sas_api_url_https'                       => 0,
			'sas_route_vendor_property_main_featured' => 'property',
			'sas_guest_panel_route'                   => 'user',
			'sas_page_home'                           => false,
			'sas_page_404'                            => false,
			'sas_one_property_page_id'                => 0,
			'sas_one_property_page_flag'              => 0,
			'sas_post_update_flag'                    => 1,
			'sas_version_save'                        => 0,
			'recovery'                                => 'recovery',
		];
	}

	public function all( $params = [] ) {
		$params = array_merge( [
			'success' => false,
			'only'    => [],
			'except'  => [],
		], $params );

		$this->settings            = [ 'data' => [] ];
		$this->settings['success'] = $params['success'];
		foreach ( self::defaults() as $key => $value ) {
			if ( in_array( $key, $params['except'] ) ) {
				continue;
			}
			if ( ! empty( $params['only'] ) and ! in_array( $key, $params['only'] ) ) {
				continue;
			}
			$this->settings['data'][ $key ] = self::get( $key, $value );
		}
		$this->sasSettingsDTO = new SasSettingsDTO( $this->settings['data'] );

        $this->retrievePropertyList($this->sasSettingsDTO);
        $this->settings['data']['properties'] 				= $this->propertyListDTO->properties;
		$this->settings['data']['guest_page_url']           = $this->retrieveGuestPageUrl();
		$this->settings['data']['sas_post_update_flag']     = $this->getSasPostUpdateFlag();
		$this->settings['data']['vendor_property_page_url'] = $this->retrieveVendorPropertyPageUrl();
		$this->settings['data']['site_url'] 				= $this->retrieveSiteUrl();

        return $this->settings;
    }

	public function getSasSettingsDTO(): SasSettingsDTO {
		return $this->sasSettingsDTO;
	}

	public function get( $key, $default = null ) {
		if ( $default === null ) {
			$defaults = self::defaults();
			if ( isset( $defaults[ $key ] ) ) {
				$default = $defaults[ $key ];
			}
		}

		return get_option( $key, $default );
	}

	function getFullUrl(): string {
		return $this->get( 'sas_api_url', '' );
	}

	function retrieveGuestPageUrl(): string {
		return get_site_url() . "/" . $this->sasSettingsDTO->sas_guest_panel_route;
	}

	function getSasPostUpdateFlag(): string {
		return $this->get( 'sas_post_update_flag', 1 );
	}

	function getSasVersionSave(): string {
		return $this->get( 'sas_version_save', 0 );
	}

	function retrieveVendorPropertyPageUrl(): string {
		return get_site_url() . "/" . $this->sasSettingsDTO->sas_route_vendor_property_main_featured;
	}

	function retrieveSiteUrl(): string {
		return get_site_url();
	}

	function retrievePropertyList( SasSettingsDTO $sasSettingsDTO ) {
		$url = $sasSettingsDTO->sas_api_url . "/api/properties/external?page=1&per_page=5";
		
		$args = [
			'headers' => [
				'Accept' => 'application/json',
				'Content-Type' => 'application/json',
				'SearchAndStayApiToken' => $sasSettingsDTO->sas_api_token,
				'User-Agent' => 'WordPress'
			],
			'timeout' => 30,
		];
		
		$response = wp_remote_get( $url, $args );
		
		if ( is_wp_error( $response ) ) {
			$error_message = $response->get_error_message();
			wp_admin_notice( $error_message, [
				'dismissible' => true,
			] );
		} else {
			$body = wp_remote_retrieve_body( $response );
			$result = json_decode( $body, true ) ?? [];
			$this->propertyListDTO = new PropertyListDTO( $result['data'] ?? [] );
		}
		
		return $this;
	}

	public function getPropertyList() {
		return $this->settings['data']['properties'] ?? [];
	}
}
